globals
[
  grid-x-inc               ;; the amount of patches in between two roads in the x direction
  grid-y-inc               ;; the amount of patches in between two roads in the y direction
  acceleration             ;; the constant that controls how much a car speeds up or slows down by if
                           ;; it is to accelerate or decelerate
  phase                    ;; keeps track of the phase
  clock                    ;; keeps track of the total times thru the go procedure
  num-cars-stopped         ;; the number of cars that are stopped during a single pass thru the go procedure
  old-display-which-metric ;; holds the value of display-which-metric for the last time through the go procedure
  current-light            ;; the currently selected light

  ;; patch agentsets
  intersections ;; agentset containing the patches that are intersections
  roads         ;; agentset containing the patches that are roads

  ;; list variables that hold data for each plot
  wait-data     ;; list that holds the average wait time of the cars for each pass through the go procedure
  stopped-data  ;; list that holds the number of stopped of the cars for each pass through the go procedure
  speed-data    ;; list that holds the average speed of the cars for each pass through the go procedure
]

turtles-own
[
  speed     ;; the speed of the turtle
  up-car?   ;; true if the turtle moves downwards and false if it moves to the right
  wait-time ;; the amount of time since the last time a turtle has moved
]

patches-own
[
  intersection?   ;; true if the patch is at the intersection of two roads
  accident?       ;; true if a crash has occurred at this intersection.
                  ;; false for non-intersection patches.
  green-light-up? ;; true if the green light is above the intersection.  otherwise, false.
                  ;; false for a non-intersection patches.
  my-row          ;; the row of the intersection counting from the upper left corner of the
                  ;; graphics window.  -1 for non-intersection patches.
  my-column       ;; the column of the intersection counting from the upper left corner of the
                  ;; graphics window.  -1 for non-intersection patches.
  my-phase        ;; the phase for the intersection.  -1 for non-intersection patches.
  auto?           ;; whether or not this intersection will switch automatically.
                  ;; false for non-intersection patches.
]


;;;;;;;;;;;;;;;;;;;;;;
;; Setup Procedures ;;
;;;;;;;;;;;;;;;;;;;;;;

;; Initialize the display by giving the global and patch variables intial values.
;; Create num-cars of turtles if there are enough road patches for one turtle to
;; be created per road patch. Set up the plots.
to setup
  ca
  setup-globals

  ;; First we ask the patches to draw themselves and set up a few variables
  setup-patches
  make-current one-of intersections
  label-current

  set-default-shape turtles "car"

  if (num-cars > count roads)
  [
    user-message "There are too many cars for the amount of " +
    "road.  Either increase the amount of roads " +
    "by increasing the GRID-SIZE-X or " +
    "GRID-SIZE-Y sliders, or decrease the " +
    "number of cars by lowering the NUMBER slider.\n" +
    "The setup has stopped."
    stop
  ]

  ;; Now create the turtles and have each created turtle call the functions setup-cars and set-car-color
  cct num-cars
  [
    setup-cars
    set-car-color
    record-data
  ]

  ;; give the turtles an initial speed
  ask turtles [ set-car-speed ]

  update-list-info
  setup-plots
end

;; Initialize the global variables to appropriate values
to setup-globals
  set current-light nobody ;; just for now, since there are no lights yet
  set phase 0
  set clock 0
  set num-cars-stopped 0
  set grid-x-inc screen-size-x / grid-size-x
  set grid-y-inc screen-size-y / grid-size-y

  ;; initialize the lists for changing the plots
  set wait-data []
  set stopped-data []
  set speed-data []

  ;; don't make acceleration 0.1 since we could get a rounding error and end up on a patch boundary
  set acceleration 0.099
end

;; Make the patches have appropriate colors, set up the roads and intersections agentsets,
;; and initialize the traffic lights to one setting
to setup-patches
  ;; initialize the patch-owned variables and color the patches to a base-color
  ask patches
  [
    set intersection? false
    set auto? false
    set accident? false
    set green-light-up? true
    set my-row -1
    set my-column -1
    set my-phase -1
    set pcolor brown + 3
  ]

  ;; initialize the global variables that hold patch agentsets
  set roads patches with
    [(floor((pxcor + screen-edge-x - floor(grid-x-inc - 1)) mod grid-x-inc) = 0) or
    (floor((pycor + screen-edge-y) mod grid-y-inc) = 0)]
  set intersections roads with
    [(floor((pxcor + screen-edge-x - floor(grid-x-inc - 1)) mod grid-x-inc) = 0) and
    (floor((pycor + screen-edge-y) mod grid-y-inc) = 0)]

  ask roads [ set pcolor white ]
  setup-intersections
end

;; Give the intersections appropriate values for the intersection?, my-row, and my-column 
;; patch variables.  Make all the traffic lights start off so that the lights are red 
;; horizontally and green vertically.  
to setup-intersections
  ask intersections
  [
    set intersection? true
    set green-light-up? true
    set my-phase 0
    set auto? true
    set my-row floor((pycor + screen-edge-y) / grid-y-inc)
    set my-column floor((pxcor + screen-edge-x) / grid-x-inc)
    set-signal-colors
  ]
end

;; Initialize the turtle variables to appropriate values and place the turtle on an empty road patch.
to setup-cars  ;; turtle procedure
  set speed 0
  set wait-time 0
  put-on-empty-road
  ifelse intersection?
  [
    ifelse random 2 = 0
    [ set up-car? true ]
    [ set up-car? false ]
  ]
  [
    ifelse (floor((pxcor + screen-edge-x - floor(grid-x-inc - 1)) mod grid-x-inc) = 0)
    [ set up-car? true ]
    [ set up-car? false ]
  ]
  ifelse up-car?
  [ set heading 180 ]
  [ set heading 90 ]
end

;; Find a road patch without any turtles on it and place the turtle there.
to put-on-empty-road  ;; turtle procedure
  let road random-one-of roads
  setxy (pxcor-of road) (pycor-of road)
  if any? other-turtles-here
  [ put-on-empty-road ]
end

;; Initialize the plots
to setup-plots
  set-current-plot "Stopped Cars"
  set-plot-y-range 0 num-cars
  plot num-cars-stopped
  
  set-current-plot "Average Wait Time of Cars"
  plot mean values-from turtles [ wait-time ]
  
  set-current-plot "Average Speed of Cars"
  set-plot-y-range 0 speed-limit
  plot mean values-from turtles [ speed ]
end


;;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Procedures ;;
;;;;;;;;;;;;;;;;;;;;;;;;

;; Run the simulation
to go
  ;; clear any accidents from the last time thru the go procedure
  clear-accidents
  update-current

  ;; have the intersections change their color
  set-signals
  set num-cars-stopped 0

  ;; set the turtles speed for this time thru the procedure, move them forward their speed,
  ;; record data for plotting, and set the color of the turtles to an appropriate color
  ;; based on their speed
  ask turtles
  [
    set-car-speed
    fd speed
    record-data
    set-car-color
  ]
  ;; crash the cars if crash? is true
  if crash?
  [ crash-cars ]

  ;; update the information in the lists for plotting
  update-list-info

  ;; update the plots with the new information from this pass thru the procedure
  do-plotting
  ;; update the clock and the phase
  clock-tick
end

to choose-current
  if mouse-down?
  [
    let x-mouse (round mouse-xcor)
    let y-mouse (round mouse-ycor)
    if intersection?-of patch-at x-mouse y-mouse
    [
      update-current
      unlabel-current
      make-current patch-at x-mouse y-mouse
      label-current
      stop
    ]
  ]
end

;; Set up the current light and the interface to change it.
to make-current [light]
  set current-light light
  set current-phase my-phase-of current-light
  set current-auto? auto?-of current-light
end

;; update the variables for the current light
to update-current
  set my-phase-of current-light current-phase
  set auto?-of current-light current-auto?
end

;; label the current light
to label-current
  ask current-light
  [
    ask patch-at -1 1
    [
      set plabel-color black
      set plabel "current"
    ]
  ]
end

;; unlabel the current light (because we've chosen a new one)
to unlabel-current
  ask current-light
  [
    ask patch-at -1 1
    [
      set plabel no-label
    ]
  ]
end

;; have the traffic lights change color if phase equals each intersections' my-phase
to set-signals
  ask intersections with [auto? and phase = floor ((my-phase * ticks-per-cycle) / 100)]
  [
    set green-light-up? (not green-light-up?)
    set-signal-colors
  ]
end

;; This procedure checks the variable green-light-up? at each intersection and sets the
;; traffic lights to have the green light up or the green light to the left.
to set-signal-colors  ;; intersection (patch) procedure
  ifelse power?
  [
    ifelse green-light-up?
    [
      set (pcolor-of patch-at -1 0) red
      set (pcolor-of patch-at 0 1) green
    ]
    [
      set (pcolor-of patch-at -1 0) green
      set (pcolor-of patch-at 0 1) red
    ]
  ]
  [
    set (pcolor-of patch-at -1 0) white
    set (pcolor-of patch-at 0 1) white
  ]
end

;; set any intersection's color that had an accident back to white and make accident? false
to clear-accidents
  if crash?
  [
    ask patches with [ accident? ]
    [
      set pcolor white
      set accident? false
    ]
  ]
end

;; set the turtles' speed based on whether they are at a red traffic light or the speed of the
;; turtle (if any) on the patch in front of them
to set-car-speed  ;; turtle procedure
  ifelse pcolor = red
  [ set speed 0 ]
  [
    ifelse up-car?
    [ set-speed 0 -1 ]
    [ set-speed 1 0 ]
  ]
end

;; set the speed variable of the turtle to an appropriate value (not exceeding the
;; speed limit) based on whether there are turtles on the patch in front of the turtle
to set-speed [ delta-x delta-y ]  ;; turtle procedure
  ;; get the turtles on the patch in front of the turtle
  let turtles-ahead turtles-at delta-x delta-y

  ;; if there are turtles in front of the turtle, slow down
  ;; otherwise, speed up
  ifelse any? turtles-ahead
  [
    let up-cars?-ahead values-from turtles-ahead [ up-car? ]
    ifelse member? up-car? up-cars?-ahead and member? (not up-car?) up-cars?-ahead
    [
      if not crash?
      [ set speed 0 ]
    ]
    [
      set speed speed-of one-of turtles-ahead
      slow-down
    ]
  ]
  [ speed-up ]
end

;; decrease the speed of the turtle
to slow-down  ;; turtle procedure
  ifelse speed <= 0  ;;if speed < 0
  [ set speed 0 ]
  [ set speed speed - acceleration ]
end

;; increase the speed of the turtle
to speed-up  ;; turtle procedure
  ifelse speed > speed-limit
  [ set speed speed-limit ]
  [ set speed speed + acceleration ]
end

;; set the color of the turtle to a different color based on how fast the turtle is moving
to set-car-color  ;; turtle procedure
  ifelse speed < (speed-limit / 2)
  [ set color blue ]
  [ set color cyan - 2 ]
end

;; keep track of the number of stopped turtles and the amount of time a turtle has been stopped 
;; if its speed is 0
to record-data  ;; turtle procedure
  ifelse speed = 0
  [
    set num-cars-stopped num-cars-stopped + 1
    set wait-time wait-time + 1
  ]
  [ set wait-time 0 ]
end

;; crash any turtles at the same intersection going in different directions
to crash-cars
  ask intersections with [ any? turtles-here with [ up-car? ] and any? turtles-here with [ not up-car? ] ]
  [
    set accident? true
    set pcolor orange
  ]
end

;; add the new information from this pass thru the go procedure to the HubNet lists
to update-list-info
  set wait-data lput ( mean values-from turtles [ wait-time ] ) wait-data
  set stopped-data lput num-cars-stopped stopped-data
  set speed-data lput ( mean values-from turtles [ speed ] ) speed-data
end

to change-current
  ask current-light
  [
    set green-light-up? (not green-light-up?)
    set-signal-colors
  ]
end

;; increases the clock by 1 and cycles phase to the next appropriate value
to clock-tick
  set clock clock + 1
  ;; The phase cycles from 0 to ticks-per-cycle, then starts over.
  set phase phase + 1
  if phase mod ticks-per-cycle = 0
  [ set phase 0 ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;
;; Plotting Procedures ;;
;;;;;;;;;;;;;;;;;;;;;;;;;

;; plot the data from this pass thru the go procedure
to do-plotting
  ifelse display-which-metric = old-display-which-metric
  [
    ;; we only need to plot 1 value since the current plot is the same as the plot we are supposed to plot to now
    ifelse display-which-metric = "stopped cars"
    [ plot-new-value "Stopped Cars" num-cars-stopped ]
    [
      ifelse display-which-metric = "average speed"
      [ plot-new-value "Average Speed of Cars" mean values-from turtles [ speed ] ]
      [
        ifelse display-which-metric = "average wait"
        [ plot-new-value "Average Wait Time of Cars" mean values-from turtles [ wait-time ] ]
        [
          plot-new-value "Stopped Cars" num-cars-stopped
          plot-new-value "Average Wait Time of Cars" mean values-from turtles [ wait-time ]
          plot-new-value "Average Speed of Cars" mean values-from turtles [ speed ]
        ]
      ]
    ]
  ]
  [
    ;; otherwise, we need to plot at least 1 list since the plot we are supposed to plot to is different from the plot we last plotted in
    ifelse display-which-metric = "stopped cars"
    [ clear-plots-and-plot-in-new-plot "Stopped Cars" stopped-data ]
    [
      ifelse display-which-metric = "average speed"
      [ clear-plots-and-plot-in-new-plot "Average Speed of Cars" speed-data ]
      [
        ifelse display-which-metric = "average wait"
        [ clear-plots-and-plot-in-new-plot "Average Wait Time of Cars" wait-data ]
        [
          ifelse old-display-which-metric = "stopped cars"
          [ plot-value-and-lists "Stopped Cars" num-cars-stopped "Average Speed of Cars" speed-data "Average Wait Time of Cars" wait-data ]
          [
            ifelse old-display-which-metric = "average speed"
            [ plot-value-and-lists "Average Speed of Cars" (mean values-from turtles [ speed ]) "Stopped Cars" stopped-data "Average Wait Time of Cars" wait-data ]
            [ plot-value-and-lists "Average Wait Time of Cars" (mean values-from turtles [ wait-time ]) "Stopped Cars" stopped-data "Average Speed of Cars" speed-data ]
          ]
        ]
      ]
    ]
    set old-display-which-metric display-which-metric
  ]
end

to plot-new-value [ name-of-plot value ]
  set-current-plot name-of-plot
  plot value
end

to clear-plots-and-plot-in-new-plot [ name-of-plot list-to-plot ]
  clear-all-plots
  plot-new-list name-of-plot list-to-plot
end

to plot-new-list [ name-of-plot list-to-plot ]
  set-current-plot name-of-plot
  clear-plot
  foreach list-to-plot
    [ plot ? ]
end

to plot-value-and-lists [ value-plot value list-plot1 list-to-plot1 list-plot2 list-to-plot2 ]
  plot-new-value value-plot value
  plot-new-list list-plot1 list-to-plot1
  plot-new-list list-plot2 list-to-plot2
end

;; update the plots to show the current value of display-which-metric
to show-current-metric-in-plots
  if display-which-metric != old-display-which-metric
  [
    ;; otherwise, we need to plot at least 1 list since the plot we are supposed to plot to is different from the plot we last plotted in
    ifelse display-which-metric = "stopped cars"
    [ clear-plots-and-plot-in-new-plot "Stopped Cars" stopped-data ]
    [
      ifelse display-which-metric = "average speed"
      [ clear-plots-and-plot-in-new-plot "Average Speed of Cars" speed-data ]
      [
        ifelse display-which-metric = "average wait"
        [ clear-plots-and-plot-in-new-plot "Average Wait Time of Cars" wait-data ]
        [
          ifelse old-display-which-metric = "stopped cars"
          [
            plot-new-list "Average Speed of Cars" speed-data
            plot-new-list "Average Wait Time of Cars" wait-data
          ]
          [
            ifelse old-display-which-metric = "average speed"
            [
              plot-new-list "Stopped Cars" stopped-data
              plot-new-list "Average Wait Time of Cars" wait-data
            ]
            [
              plot-new-list "Stopped Cars" stopped-data
              plot-new-list "Average Speed of Cars" speed-data
            ]
          ]
        ]
      ]
    ]
    set old-display-which-metric display-which-metric
  ]
end


; *** NetLogo Model Copyright Notice ***
;
; This model was created as part of the projects:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN
; CLASSROOMS and INTEGRATED SIMULATION AND MODELING ENVIRONMENT.
; The project gratefully acknowledges the support of the
; National Science Foundation (REPP & ROLE programs) -- grant numbers 
; REC #9814682 and REC-0126227.
;
; Copyright 2003 by Uri Wilensky.  Updated 2003.  All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from Uri Wilensky.
; Contact Uri Wilensky for appropriate licenses for redistribution for
; profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (2003).  NetLogo Gridlock model.
; http://ccl.northwestern.edu/netlogo/models/Gridlock.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 1998 by Uri Wilensky.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/Gridlock
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
354
10
697
374
18
18
9.0
1
12
1
1
1
0

CC-WINDOW
5
583
749
678
Command Center

PLOT
491
380
740
569
Average Wait Time of Cars
Time
Average Wait
0.0
100.0
0.0
5.0
true
false
PENS
"default" 1.0 0 -65536 true

PLOT
241
380
490
569
Average Speed of Cars
Time
Average Speed
0.0
100.0
0.0
1.0
true
false
PENS
"default" 1.0 0 -65536 true

SLIDER
100
10
197
43
grid-size-y
grid-size-y
1
9
5
1
1
NIL

SLIDER
4
10
98
43
grid-size-x
grid-size-x
1
9
5
1
1
NIL

SWITCH
101
82
191
115
crash?
crash?
1
1
-1000

SWITCH
4
82
99
115
power?
power?
0
1
-1000

SLIDER
4
46
285
79
num-cars
num-cars
0
400
200
1
1
NIL

PLOT
6
380
240
569
Stopped Cars
Time
Stopped Cars
0.0
100.0
0.0
100.0
true
false
PENS
"default" 1.0 0 -65536 true

BUTTON
208
134
272
167
Go
go
T
1
T
OBSERVER
T
NIL

BUTTON
200
10
284
43
Setup
setup
NIL
1
T
OBSERVER
T
NIL

SLIDER
3
152
157
185
speed-limit
speed-limit
0.0
1.0
1.0
0.1
1
NIL

MONITOR
192
82
297
131
Current Phase
phase
3
1

SLIDER
3
118
157
151
ticks-per-cycle
ticks-per-cycle
1
100
20
1
1
NIL

BUTTON
175
335
314
380
Show Current Plot
show-current-metric-in-plots
NIL
1
T
OBSERVER
T
NIL

SLIDER
140
186
296
219
current-phase
current-phase
0
99
0
1
1
%

BUTTON
3
222
137
255
Change light
change-current
NIL
1
T
OBSERVER
T
NIL

SWITCH
3
186
138
219
current-auto?
current-auto?
0
1
-1000

BUTTON
139
222
294
255
Select intersection
choose-current
T
1
T
OBSERVER
NIL
NIL

CHOOSER
4
335
173
380
display-which-metric
display-which-metric
"stopped cars" "average speed" "average wait" "all plots"
3

@#$#@#$#@
WHAT IS IT?
-----------
You control traffic lights and overall variables, such as the speed limit and the number of cars, in a real-time traffic simulation.  This allows you to explore traffic dynamics, which can lead into many areas of study, from calculus to social studies.

Try to develop strategies to improve traffic and to understand the different ways to measure the quality of traffic.


HOW IT WORKS
------------
Everything in this model is synchronized to a global clock.  Each time step, the cars simply attempt to move forward at their current speed.  If their current speed is less than the speed limit and there is no car directly in front of them, they accelerate.  If there is a slower car in front of them, they match the speed of the slower car and declerate.  If there is a red light or a stopped car in front of them, they stop. 

There are two different ways the lights can change.  First, the user can change any light at any time by making the light current, and then clicking CHANGE LIGHT.  Second, lights can change automatically, once per cycle.  Initially, all lights will automatically change at the beginning of each cycle.


HOW TO USE IT
-------------
Change the traffic grid (using the sliders GRID-SIZE-X and GRID-SIZE-Y) to make the desired number of lights.  Change any other of the settings that you would like to change.  Press the SETUP button.

At this time, you may configure the lights however you like, with any combination of auto/manual and any phase. Changes to the state of the current light are made using the CURRENT-AUTO?, CURRENT-PHASE and CHANGE LIGHT controls.  You may select the current intersection using the SELECT INTERSECTION control.  See below for details.

Start the simulation by pressing the GO button.  You may continue to make changes to the lights while the simulation is running.

BUTTONS:
  SETUP - generates a new traffic grid based on the current GRID-SIZE-X and GRID-SIZE-Y and NUM-CARS number of cars.  This also clears all the plots. All lights are set to auto, and all phases are set to 0.
  GO - runs the simulation indefinitely
  CHANGE LIGHT - changes the direction traffic may flow through the current light. A light can be changed manually even if it is operating in auto mode.
  SELECT INTERSECTION - allows you to select a new "current" light. When this button is depressed, click in the intersection which you would like to make current. When you've selected an intersection, the "current" label will move to the new intersection and this button will automatically pop up.
  SHOW CURRENT PLOT - displays the plot information for the plot currently selected by the DISPLAY-WHICH-METRIC chooser

SLIDERS:
  SPEED-LIMIT - sets the maximum speed for the cars
  NUM-CARS - the number of cars in the simulation (you must press the SETUP button to see the change)
  TICKS-PER-CYCLE - sets the number of ticks that will elapse for each cycle.  This has no effect on manual lights.  This allows you to increase or decrease the granularity with which lights can automatically change.
  GRID-SIZE-X - sets the number of vertical roads there are (you must press the SETUP button to see the change)
  GRID-SIZE-Y - sets the number of horizontal roads there are (you must press the SETUP button to see the change)
  CURRENT-PHASE - controls when the current light changes, if it is in auto mode. The slider value represents the percentage of the way through each cycle at which the light should change. So, if the TICKS-PER-CYCLE is 20 and CURRENT-PHASE is 75%, the current light will switch at tick 15 of each cycle.

SWITCHES:
  CRASH? - toggles car crashing
  POWER? - toggles the presence of traffic lights
  CURRENT-AUTO? - toggles the current light between automatic mode, where it changes once per cycle (according to CURRENT-PHASE), and manual, in which you directly control it with CHANGE LIGHT.

CHOOSERS:
  DISPLAY-WHICH-METRIC - determines which plot is drawn in NetLogo

PLOTS:
  STOPPED CARS - displays the number of stopped cars over time
  AVERAGE SPEED OF CARS - displays the average speed of cars over time
  AVERAGE WAIT TIME OF CARS - displays the average time cars are stopped over time


THINGS TO NOTICE
----------------
When cars have stopped at a traffic light, and then they start moving again, the traffic jam will move backwards even though the cars are moving forwards.  Why is this?


THINGS TO TRY
-------------
Try changing the speed limit for the cars.  How does this affect the overall efficiency of the traffic flow?  Are fewer cars stopping for a shorter amount of time?  Is the average speed of the cars higher or lower than before?

Try changing the number of cars on the roads.  Does this affect the efficiency of the traffic flow?

How about changing the speed of the simulation?  Does this affect the efficiency of the traffic flow?

Try running this simulation with all lights automatic.  Is it harder to make the traffic move well using this scheme than controlling one light manually?  Why?

Try running this simulation with all lights automatic.  Try to find a way of setting the phases of the traffic lights so that the average speed of the cars is the highest.  Now try to minimize the number of stopped cars.  Now try to decrease the average wait time of the cars.  Is there any correlation between these different metrics?


EXTENDING THE MODEL
-------------------
Currently, the maximum speed limit (found in the SPEED-LIMIT slider) for the cars is 1.0.  This is due to the fact that the cars must look ahead the speed that they are traveling to see if there are cars ahead of them.  If there aren't, they speed up.  If there are, they slow down.  Looking ahead for a value greater than 1 is a little bit tricky.  Try implementing the correct behavior for speeds greater than 1.


NETLOGO FEATURES
----------------
This model uses two forever buttons which may run at the same time, to allow the user to select a new current intersection while the model is running.

It also uses a chooser to allow the user to choose between several different possible plots, or to display all of them at once.


RELATED MODELS
--------------
Traffic Basic, which simulates the flow of a single lane of traffic in one direction.

Gridlock for calculator HubNet and Gridlock for computer HubNet, which have very similar functionality, but allow a group of users to control the cars in a participatory fashion.


CREDITS AND REFERENCES
----------------------
To refer to this model in academic publications, please use: Wilensky, U. (2002). NetLogo Gridlock model. http://ccl.northwestern.edu/netlogo/models/Gridlock. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 2002 by Uri Wilensky.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/Gridlock for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
true
0
Polygon -7566196 true true 180 15 164 21 144 39 135 60 132 74 106 87 84 97 63 115 50 141 50 165 60 225 150 285 165 285 225 285 225 15 180 15
Circle -16777216 true false 180 30 90
Circle -16777216 true false 180 180 90
Polygon -16777216 true false 80 138 78 168 135 166 135 91 105 106 96 111 89 120
Circle -7566196 true true 195 195 58
Circle -7566196 true true 195 47 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
